<?php
require dirname(__FILE__) . '/../../TestUnitDB.php';

/**
 * Tests core API functionality via CURL
 */
class CurlApiTest extends TestUnitDB
{
    protected $_obejct;

    protected function setUp()
    {
        parent::setUp();
    }

    protected function tearDown()
    {
    }

    /**
     * Sample REST client
     *
     * @param  string $rest_path REST resource name
     * @param  string $method    HTTM Method
     * @param  mixed  $params    POST/PUT params
     * @param  array  $headers   Headers
     * @param  array  $auth      Auth params (username, apy key)
     * @return array  Response info, response body
     */
    protected function sendRequest($rest_path, $method, $params, $headers, $auth = array())
    {
        $curl = curl_init(Registry::get('config.http_location') . "/api.php" . "?" . http_build_query(array('q' => $rest_path)));

        curl_setopt($curl, CURLOPT_HEADER, 0);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);

        if (!empty($auth)) {
            curl_setopt($curl, CURLOPT_HTTPAUTH, CURLAUTH_BASIC ) ;
            curl_setopt($curl, CURLOPT_USERPWD, $auth[0] . ":" . $auth[1]);
        }

        if ($method == 'PUT') {
            $data = (is_array($params)) ? http_build_query($params) : $params;
            curl_setopt($curl, CURLOPT_CUSTOMREQUEST, $method);
            curl_setopt($curl, CURLOPT_POSTFIELDS, $data);
        } elseif ($method == 'POST') {
            $data = (is_array($params)) ? http_build_query($params) : $params;
            curl_setopt($curl, CURLOPT_POST, 1);
            curl_setopt($curl, CURLOPT_POSTFIELDS, $data);
        } else {
            curl_setopt($curl, CURLOPT_CUSTOMREQUEST, $method);
        }

        curl_setopt($curl, CURLOPT_HTTPHEADER, $headers);

        $result = curl_exec($curl);
        $info = curl_getinfo($curl);

        curl_close($curl);

        return array($info, $result);
    }

    /**
     * API auth tests.
     *
     * @dataProvider authProvider()
     * @datasetProvider usersDatset
     */
    public function testApiAuth($rest_path, $method, $params, $user_login, $api_key, $response_status)
    {
        list($info, $result) = $this->sendRequest($rest_path, $method, $params, array(), array($user_login, $api_key));

        if ($info['http_code'] != $response_status) {
            print_r($result);
        }

        $this->assertEquals($info['http_code'], $response_status);
    }

    public function authProvider()
    {
        $api_key =  Api::generateKey();

        return array(
            array(
                // Request with empty auth data
                "users/1", "GET", array(), '', '',
                // Response
                401
            ),
            array(
                // Request with empty email data and right key
                "users/1", "GET", array(), '', $api_key,
                // Response
                401
            ),
            array(
                // Request with right email and empty key
                "users/1", "GET", array(), 'admin@example.com', '',
                // Response
                401
            ),
            array(
                // Request with right email and wrong key
                "users/1", "GET", array(), 'admin@example.com', $api_key,
                // Response
                401
            ),
            array(
                // Request not existing resource
                "users/1", "GET", array(), 'admin@example.com', 'asdksuhd87ahdiu34h8dq7hildjhc78qahw4iuhdaskjfhliah4783fh8347fyh87q34hfklj',
                // Response
                401
            ),
            array(
                // Request with right email and right key
                "users/1", "GET", array(), 'admin@example.com', '4C1Z794Y3KumSIt15HTq3Db87fhaz68v',
                // Response
                200
            ),
            // Test injections
            array(
                // Request with empty auth data
                "users/1", "GET", array(), "' OR 1=1 #", "' OR 1=1 #",
                // Response
                401
            ),
            array(
                // Request with empty auth data
                "users/1", "GET", array(), "' OR 1=1 /*", "' OR 1=1 /*",
                // Response
                401
            ),
            array(
                // Request with empty auth data
                "users/1", "GET", array(), "' OR 1=1 --", "' OR 1=1 --",
                // Response
                401
            ),
        );
    }

    /**
     * Tests that api returns wright response codes on different rerquests
     *
     * @dataProvider responseCodesProvider()
     * @datasetProvider usersDatset
     */
    public function testApiResponseCodes($rest_path, $method, $params, $headers, $response_status)
    {
        $api_key =  '4C1Z794Y3KumSIt15HTq3Db87fhaz68v';

        list($info, $result) = $this->sendRequest($rest_path, $method, $params, $headers,  array('admin@example.com', $api_key));

        if ($info['http_code'] != $response_status) {
            print_r($result);
        }

        $this->assertEquals($info['http_code'], $response_status);
    }

    public function responseCodesProvider()
    {
        return array(
            array(
                // Request not existing resource
                "", "GET", array(), array('Content-type: application/json'),
                // Response
                404
            ),
            array(
                // Request not existing resource
                "", "POST", array(), array('Content-type: application/json'),
                // Response
                404
            ),
            array(
                // Request not existing resource
                "", "PUT", array(), array('Content-type: application/json'),
                // Response
                404
            ),
            array(
                // Request not existing resource
                "", "DELETE", array(), array('Content-type: application/json'),
                // Response
                404
            ),
            array(
                // Try to get resource with wrong Content-type
                "users", "GET", array(), array('Content-type: application/xml'),
                // Response
                415
            ),
            array(
                // Try to get resource with not existed HTTP method
                "users", "AAAA", array(), array('Content-type: application/json'),
                // Response
                405
            ),
            array(
                // Try to get info about resource
                "users/1", "GET", array(), array('Content-type: application/json'),
                // Response
                200
            ),
            array(
                // Try to get items list
                "users", "GET", array(), array('Content-type: application/json'),
                // Response
                200
            ),
            array(
                // Try to create resource with empty data
                "users", "POST", array(), array('Content-type: application/json'),
                // Response
                405
            ),
            array(
                // Try to create resource with data
                "users", "POST", array('username' => 'test', 'status' => 'D'), array('Content-type: text/plain'),
                // Response
                201
            ),
            array(
                // Try to get create resource with id. Update by POST not allowed
                "users/1", "POST", array('name' => 'test'), array('Content-type: text/plain'),
                // Response
                405
            ),
            array(
                // try to delete without id
                "users", "DELETE", array(), array('Content-type: text/plain'),
                // Response
                405
            ),
            array(
                // Try to delete
                "users/3", "DELETE", array(), array('Content-type: text/plain'),
                // Response
                204
            ),
            array(
                // Try to create resource with PUT
                "users", "PUT", array('username' => 'test', 'status' => 'D'), array('Content-type: text/plain'),
                // Response
                405
            ),
            array(
                // Try to update resource
                "users/1", "PUT", array('username' => 'test', 'status' => 'D'), array('Content-type: text/plain'),
                // Response
                200
            ),
            array(
                // Try to update with empty data
                "users/1", "PUT", array(), array('Content-type: application/json'),
                // Response
                405
            ),
        );
    }

    protected function usersDatset()
    {
        return array(
            'users.xml',
        );
    }
}
